const http = require('http');
const { WebSocketServer } = require('ws');
const url = require('url');
const admin = require('firebase-admin');
const fetch = require('node-fetch'); // Pour appeler Laravel

// ==========================================================
// 1. CONFIGURATION
// ==========================================================

const LARAVEL_API_URL = 'https://cdn-aboapp.online/api'; // MODIFIER SI NÉCESSAIRE (/api ou /public/api selon votre config)

try {
  const serviceAccount = require('./serviceAccountKey.json');
  admin.initializeApp({
    credential: admin.credential.cert(serviceAccount)
  });
  console.log('[INIT] Firebase Admin initialisé avec succès.');
} catch (e) {
  console.error('[INIT] ERREUR: Fichier serviceAccountKey.json introuvable. Veuillez l\'uploader.');
  process.exit(1);
}

// ==========================================================
// 2. GESTION DES SALLES ET CLIENTS
// ==========================================================

const rooms = new Map(); // Stocke les salles (ex: "battle-48" -> Set[ws1, ws2])
const clientData = new Map(); // Stocke les données d'un client (ex: ws -> {userId: 10, room: "battle-48", token: "..."})

// ==========================================================
// 3. FONCTIONS UTILITAIRES
// ==========================================================

/**
 * Valide une action et ses données
 */
function validateAction(action, data) {
  switch (action) {
    case 'find_ranked_battle':
      return { valid: true };
    
    case 'create_friendly_battle':
      return { valid: true };
    
    case 'join_friendly_battle':
      if (!data.battleId || typeof data.battleId !== 'number') {
        return { valid: false, error: 'Invalid battle ID' };
      }
      return { valid: true };
    
    case 'answer_question':
      if (!data.battle_id || typeof data.battle_id !== 'number') {
        return { valid: false, error: 'Invalid battle_id' };
      }
      if (data.question_index === undefined || typeof data.question_index !== 'number') {
        return { valid: false, error: 'Invalid question_index' };
      }
      if (typeof data.is_correct !== 'boolean') {
        return { valid: false, error: 'Invalid is_correct (must be boolean)' };
      }
      return { valid: true };
    
    default:
      return { valid: false, error: 'Unknown action' };
  }
}

/**
 * Fait une requête HTTP à l'API Laravel
 */
async function apiRequest(method, endpoint, token, body = null) {
  const headers = {
    'Authorization': `Bearer ${token}`,
    'Accept': 'application/json',
    'Content-Type': 'application/json'
  };
  const options = { method: method.toUpperCase(), headers: headers };
  if (body) { options.body = JSON.stringify(body); }
  
  try {
    const response = await fetch(`${LARAVEL_API_URL}${endpoint}`, options);
    const responseText = await response.text();
    
    // Gérer les 404 comme des réponses valides (pas d'erreur)
    if (response.status === 404) {
      console.log(`[API] 404 sur ${endpoint}: ${responseText}`);
      return null;
    }
    
    if (!response.ok) {
      console.error(`[ERREUR API] ${response.status} sur ${endpoint}: ${responseText}`);
      let errorMessage = responseText;
      try {
        const errorJson = JSON.parse(responseText);
        errorMessage = errorJson.message || errorJson.error || responseText;
      } catch (e) { 
        // Ce n'était pas du JSON, utiliser le texte brut
      }
      throw new Error(`Erreur API Laravel: ${response.status} - ${errorMessage}`);
    }
    
    // Réponse vide (204 No Content)
    if (response.status === 204 || responseText.trim() === '') {
      return null;
    }
    
    try {
      return JSON.parse(responseText);
    } catch (e) {
      console.error(`[ERREUR API] Réponse non-JSON: ${responseText}`);
      throw new Error('Réponse API invalide (non-JSON)');
    }
  } catch (error) {
    // Gérer les erreurs réseau
    if (error.message.includes('fetch failed') || error.message.includes('ECONNREFUSED')) {
      throw new Error('Impossible de contacter le serveur Laravel');
    }
    throw error;
  }
}

/**
 * Ajoute un client à une salle
 */
function joinRoom(ws, roomKey) {
  leaveRoom(ws); 
  if (!rooms.has(roomKey)) {
    rooms.set(roomKey, new Set());
  }
  const clientInfo = clientData.get(ws);
  if (clientInfo) {
    rooms.get(roomKey).add(ws);
    clientInfo.room = roomKey;
    console.log(`[ROOM] Client ${clientInfo.userId} ajouté à ${roomKey}.`);
  }
}

/**
 * Retire un client d'une salle
 */
function leaveRoom(ws) {
  const data = clientData.get(ws);
  if (!data || !data.room) return;
  const roomKey = data.room;
  if (rooms.has(roomKey)) {
    rooms.get(roomKey).delete(ws);
    if (rooms.get(roomKey).size === 0) {
      rooms.delete(roomKey);
      console.log(`[ROOM] Salle ${roomKey} vide, supprimée.`);
    }
  }
  if(clientData.has(ws)) {
    clientData.get(ws).room = null;
  }
}

/**
 * Diffuse un message à tous les clients d'une salle
 */
function broadcast(roomKey, data) {
  if (rooms.has(roomKey)) {
    const message = JSON.stringify(data);
    rooms.get(roomKey).forEach(client => {
      if (client.readyState === client.OPEN) {
        client.send(message);
      }
    });
  }
}

// ==========================================================
// 4. LE SERVEUR HTTP (Pour le démarrage Passenger)
// ==========================================================

const server = http.createServer((req, res) => {
  res.writeHead(200, { 'Content-Type': 'text/plain' });
  res.end('Serveur WebSocket Cadenas Master (Full Node.js) Actif.');
});

// ==========================================================
// 5. LE SERVEUR WEBSOCKET
// ==========================================================

const wss = new WebSocketServer({ server });

wss.on('connection', async (ws, req) => {
  let userId = null; // ID de l'utilisateur (de la DB Laravel)
  let userToken = null; // Token Firebase (pour les appels API)
  
  console.log('[CONNEXION] Un client tente de se connecter...');
  
  try {
    // --- A. AUTHENTIFICATION ---
    const query = url.parse(req.url, true).query;
    userToken = query.token; // Le token est envoyé dans l'URL
    
    if (!userToken) {
      throw new Error('Token Firebase manquant');
    }
    
    const decodedToken = await admin.auth().verifyIdToken(userToken);
    
    // Appeler la route GET /api/user (de votre api.php)
    const userResponse = await apiRequest('get', '/user', userToken);
    
    if (!userResponse || !userResponse.id) {
       throw new Error('Utilisateur introuvable dans Laravel');
    }
    
    userId = userResponse.id;
    
    // Stocker les infos du client (INCLUANT LE TOKEN pour la déconnexion)
    clientData.set(ws, { 
      userId: userId, 
      room: null,
      token: userToken // IMPORTANT: Stocker le token pour l'abandon de bataille
    });
    
    console.log(`[AUTH] Succès: Client authentifié (UserID: ${userId})`);
    ws.send(JSON.stringify({ _event_type: 'authenticated' }));
    
  } catch (e) {
    console.log(`[AUTH] Rejet: ${e.message}`);
    ws.close(1008, e.message);
    return;
  }
  
  // --- B. GESTION DES MESSAGES (Commandes du client) ---
  ws.on('message', async (message) => {
    let data;
    try { 
      data = JSON.parse(message.toString()); 
    } catch (e) { 
      ws.send(JSON.stringify({ _event_type: 'error', message: 'Invalid JSON' }));
      return;
    }
    
    if (!data.action) {
      ws.send(JSON.stringify({ _event_type: 'error', message: 'Missing action' }));
      return;
    }
    
    // Valider l'action
    const validation = validateAction(data.action, data);
    if (!validation.valid) {
      ws.send(JSON.stringify({ _event_type: 'error', message: validation.error }));
      return;
    }
    
    console.log(`[MESSAGE] Commande reçue de ${userId}: ${data.action}`);
    
    try {
      let battle = null;
      let roomKey = null;
      
      switch (data.action) {
        
        // --- Cas: Le client cherche une partie classée ---
        case 'find_ranked_battle':
          
          let foundBattle = null;
          try {
            // 1. D'abord, on cherche une partie (GET /api/battles/waiting)
            console.log(`[GAME] Joueur ${userId} cherche une bataille classée...`);
            foundBattle = await apiRequest('get', '/battles/waiting?mode=ranked', userToken);
          } catch(e) {
            console.log(`[GAME] Aucune bataille classée trouvée (404 attendu): ${e.message}`);
            foundBattle = null;
          }
          
          if (foundBattle && foundBattle.id) {
            // 2. Une bataille a été trouvée, on la rejoint (POST /api/battles/{id}/join)
            console.log(`[GAME] Bataille ${foundBattle.id} trouvée. Tentative de jonction...`);
            battle = await apiRequest('post', `/battles/${foundBattle.id}/join`, userToken);
            roomKey = `battle-${battle.id}`;
            joinRoom(ws, roomKey); 
            
            // 3. On diffuse l'événement à tout le monde dans la salle
            broadcast(roomKey, { _event_type: 'battle.updated', battle: battle });
            console.log(`[GAME] Joueur ${userId} a rejoint ${roomKey}. Diffusion.`);
          } else {
            // 4. Aucune bataille trouvée, on en crée une (POST /api/battles)
             console.log(`[GAME] Aucune bataille trouvée. Création d'une nouvelle bataille...`);
             battle = await apiRequest('post', '/battles', userToken, { mode: 'ranked' });
             if (battle && battle.id) {
               roomKey = `battle-${battle.id}`;
               joinRoom(ws, roomKey);
               // On informe le créateur que sa salle est prête
               ws.send(JSON.stringify({ _event_type: 'waiting_for_player', battle: battle }));
               console.log(`[GAME] Joueur ${userId} a créé ${roomKey}.`);
             }
          }
          break;
        
        // --- Cas: Le client crée une partie amicale ---
        case 'create_friendly_battle':
           battle = await apiRequest('post', '/battles', userToken, { mode: 'friendly' });
           if (battle && battle.id) {
             roomKey = `battle-${battle.id}`;
             joinRoom(ws, roomKey);
             ws.send(JSON.stringify({ _event_type: 'waiting_for_player', battle: battle }));
             console.log(`[GAME] Joueur ${userId} a créé ${roomKey}.`);
           }
           break;
        
        // --- Cas: Le client rejoint une partie amicale ---
        case 'join_friendly_battle':
          const battleId = data.battleId; // L'UI doit d'abord trouver l'ID
          if (!battleId) break;
          battle = await apiRequest('post', `/battles/${battleId}/join`, userToken);
          if (battle && battle.id) {
            roomKey = `battle-${battle.id}`;
            joinRoom(ws, roomKey);
            broadcast(roomKey, { _event_type: 'battle.updated', battle: battle });
            console.log(`[GAME] Joueur ${userId} a rejoint ${roomKey}. Diffusion.`);
          }
          break;
        
        // --- Cas: Le client répond à une question ---
        case 'answer_question':
          if (!data.battle_id || data.question_index === undefined || data.is_correct === undefined) {
            ws.send(JSON.stringify({ 
              _event_type: 'error', 
              message: 'Missing required fields: battle_id, question_index, is_correct' 
            }));
            break;
          }
          
          try {
            battle = await apiRequest('post', `/battles/${data.battle_id}/answer`, userToken, {
              question_index: data.question_index,
              is_correct: data.is_correct
            });
            
            if (battle && battle.id) {
              roomKey = `battle-${battle.id}`;
              // S'assurer que le client est dans la salle
              const clientInfo = clientData.get(ws);
              if (!clientInfo || clientInfo.room !== roomKey) {
                joinRoom(ws, roomKey);
              }
              // Diffuser la mise à jour à tous les clients dans la salle
              broadcast(roomKey, { _event_type: 'battle.updated', battle: battle });
              console.log(`[GAME] Joueur ${userId} a répondu à la question ${data.question_index} dans ${roomKey}.`);
              
              // Si la bataille est terminée, diffuser l'événement de fin
              if (battle.status === 'finished') {
                broadcast(roomKey, { _event_type: 'battle.finished', battle: battle });
                console.log(`[GAME] Bataille ${battle.id} terminée.`);
              }
            }
          } catch (e) {
            console.error(`[ERREUR ACTION] answer_question: ${e.message}`);
            ws.send(JSON.stringify({ _event_type: 'error', message: e.message }));
          }
          break;
      }
    } catch (e) {
      console.error(`[ERREUR ACTION] ${data.action}: ${e.message}`);
      ws.send(JSON.stringify({ _event_type: 'error', message: e.message }));
    }
  });
  
  // --- C. GESTION DE LA DÉCONNEXION ---
  ws.on('close', async () => {
    console.log(`[CONNEXION] Client ${userId} déconnecté.`);
    const data = clientData.get(ws);
    
    if (data && data.room) {
      // Extraire l'ID de la bataille depuis roomKey (ex: "battle-48" -> 48)
      const battleId = data.room.replace('battle-', '');
      
      try {
        // Appeler Laravel pour abandonner la bataille
        if (data.token) {
          const battle = await apiRequest('post', `/battles/${battleId}/abandon`, data.token);
          
          // Diffuser la bataille mise à jour aux autres joueurs dans la salle
          if (battle) {
            broadcast(data.room, { 
              _event_type: 'battle.updated', 
              battle: battle
            });
            
            // Si la bataille est terminée, diffuser aussi l'événement de fin
            if (battle.status === 'finished') {
              broadcast(data.room, { 
                _event_type: 'battle.finished', 
                battle: battle
              });
            }
          } else {
            // Fallback si la bataille n'est pas retournée
            broadcast(data.room, { 
              _event_type: 'battle.abandoned', 
              battle_id: battleId,
              user_id: data.userId
            });
          }
          
          console.log(`[GAME] Bataille ${battleId} abandonnée par le joueur ${data.userId}.`);
        }
      } catch (e) {
        console.error(`[ERREUR] Impossible d'abandonner la bataille ${battleId}: ${e.message}`);
        // En cas d'erreur, diffuser quand même l'abandon
        broadcast(data.room, { 
          _event_type: 'battle.abandoned', 
          battle_id: battleId,
          user_id: data.userId
        });
      }
      
      leaveRoom(ws, data.room);
    }
    
    clientData.delete(ws);
  });
});

// ==========================================================
// 6. DÉMARRAGE DU SERVEUR
// ==========================================================

const port = process.env.PORT; // Passenger fournit ce port

server.listen(port, () => {
  console.log(`Serveur WebSocket (Full Node.js v3) démarré sur le port ${port}`);
  console.log(`[CONFIG] URL API Laravel: ${LARAVEL_API_URL}`);
});
