<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Battle;
use App\Services\BattleService;
use App\Services\MatchmakingService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Log;

class BattleController extends Controller
{
    protected BattleService $battleService;
    protected MatchmakingService $matchmakingService;

    public function __construct(BattleService $battleService, MatchmakingService $matchmakingService)
    {
        $this->battleService = $battleService;
        $this->matchmakingService = $matchmakingService;
    }

    public function create(Request $request): JsonResponse
    {
        $user = $request->user();
        $mode = $request->input('mode', 'ranked');
        $roomId = $request->input('room_id');

        $battle = $this->battleService->createBattle($user, $mode, $roomId);

        return response()->json($battle, 201);
    }

    /**
     * Display the specified battle.
     * Uses Route Model Binding - Laravel automatically resolves Battle from the route parameter.
     */
    public function show(Request $request, Battle $battle): JsonResponse
    {
        $battle->load(['player1', 'player2', 'winner']);
        return response()->json($battle);
    }

    /**
     * Join a battle.
     * Uses Route Model Binding - Laravel automatically resolves Battle from the route parameter.
     */
    public function join(Request $request, Battle $battle): JsonResponse
    {
        $user = $request->user();

        try {
            $battle = $this->battleService->joinBattle($battle, $user);
            return response()->json($battle);
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 400);
        }
    }

    /**
     * Abandon a battle.
     * Uses Route Model Binding - Laravel automatically resolves Battle from the route parameter.
     */
    public function abandon(Request $request, Battle $battle): JsonResponse
    {
        $user = $request->user();

        try {
            $this->battleService->abandonBattle($battle, $user);
            // Recharger la bataille depuis la base de données pour avoir les données à jour
            $battle->refresh();
            // Retourner la bataille mise à jour pour que Node.js puisse la diffuser
            return response()->json($battle->load(['player1', 'player2', 'winner']));
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 400);
        }
    }

    /**
     * Delete a battle.
     */
    public function destroy(Request $request, Battle $battle): JsonResponse
    {
        $user = $request->user();

        // CORRECTION LOGIQUE 403:
        // L'utilisateur doit être player1 ET la bataille doit être 'waiting'
        if ($battle->player1_id == $user->id && $battle->status === 'waiting') {
             $battle->delete();
             return response()->json(['message' => 'Battle deleted by creator']);
        }
        
        // Si l'utilisateur n'est pas player1, OU si la bataille n'est pas 'waiting'
        // On ne le laisse pas supprimer, mais on ne renvoie pas d'erreur,
        // car le serveur s'occupe du nettoyage.
        Log::warning('Tentative de suppression de bataille non autorisée', [
            'user_id' => $user->id,
            'battle_id' => $battle->id,
            'player1_id' => $battle->player1_id,
            'status' => $battle->status
        ]);

        // On renvoie un 200 OK pour que l'app Flutter ne voie pas d'erreur.
        // Le serveur s'occupera de nettoyer la bataille plus tard.
        return response()->json(['message' => 'Battle state not eligible for deletion by user. Server will cleanup.']);
    }

    public function findWaiting(Request $request): JsonResponse
    {
        $user = $request->user();
        $mode = $request->input('mode', 'ranked');

        $battle = $this->matchmakingService->findWaitingBattle($user, $mode);

        if ($battle) {
            return response()->json($battle);
        }

        return response()->json(['message' => 'No waiting battle found'], 404);
    }

    public function findFriendly(Request $request, string $roomId): JsonResponse
    {
        $battle = $this->matchmakingService->findFriendlyBattle($roomId);

        if ($battle) {
            return response()->json($battle);
        }

        return response()->json(['message' => 'Battle not found'], 404);
    }

    /**
     * Answer a question in a battle.
     * Updates the score and broadcasts the update.
     */
    public function answer(Request $request, Battle $battle): JsonResponse
    {
        $user = $request->user();
        
        $request->validate([
            'question_index' => 'required|integer|min:0',
            'is_correct' => 'required|boolean',
        ]);

        try {
            $this->battleService->updateScore(
                $battle, 
                $user, 
                $request->input('question_index'), 
                $request->input('is_correct')
            );
            
            // Recharger la bataille depuis la base de données pour avoir les données à jour
            $battle->refresh();
            
            return response()->json($battle->load(['player1', 'player2', 'winner']));
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 400);
        }
    }
}
