<?php

namespace App\Http\Controllers;

use Illuminate\Routing\Controller as BaseController;
use Illuminate\Http\Request;
use App\Services\FirebaseAuthService;
use App\Models\User;
use App\Models\Battle;
use Illuminate\Support\Facades\Log;
use Pusher\Pusher;

class BroadcastingController extends BaseController
{
    protected FirebaseAuthService $firebaseAuthService;

    public function __construct(FirebaseAuthService $firebaseAuthService)
    {
        $this->firebaseAuthService = $firebaseAuthService;
    }

    /**
     * Authenticate the request for channel access.
     * This endpoint is called by Pusher when subscribing to private/presence channels.
     */
    public function authenticate(Request $request)
    {
        try {
            // Get the Firebase token from the Authorization header
            $token = $request->bearerToken() ?? $request->query('token');

            if (!$token) {
                Log::error('[BROADCAST AUTH] No token provided', [
                    'headers' => $request->headers->all(),
                    'query' => $request->query->all(),
                ]);
                return response()->json(['error' => 'Unauthorized. Token missing.'], 401);
            }

            // Verify the Firebase token
            $uid = $this->firebaseAuthService->verifyToken($token);

            if (!$uid) {
                Log::error('[BROADCAST AUTH] Invalid token', [
                    'uid' => $uid,
                    'token_preview' => substr($token, 0, 20) . '...',
                ]);
                return response()->json(['error' => 'Unauthorized. Invalid token.'], 401);
            }

            Log::info('[BROADCAST AUTH] Token verified', [
                'firebase_uid' => $uid,
            ]);

            // Get or create the user
            $user = $this->firebaseAuthService->findOrCreateUser($uid);

            if (!$user) {
                Log::error('[BROADCAST AUTH] User not found', [
                    'uid' => $uid,
                ]);
                return response()->json(['error' => 'Unauthorized. User not found.'], 401);
            }

            Log::info('[BROADCAST AUTH] User found/created', [
                'user_id' => $user->id,
                'firebase_uid' => $user->firebase_uid,
                'display_name' => $user->display_name,
            ]);

            // Get channel name and socket ID from the request
            $channelName = $request->input('channel_name');
            $socketId = $request->input('socket_id');

            if (!$channelName || !$socketId) {
                Log::error('[BROADCAST AUTH] Missing channel_name or socket_id', [
                    'channel_name' => $channelName,
                    'socket_id' => $socketId,
                    'input' => $request->all(),
                ]);
                return response()->json(['error' => 'Bad request. Missing channel_name or socket_id.'], 400);
            }

            Log::info('[BROADCAST AUTH] Channel authorization request', [
                'user_id' => $user->id,
                'channel_name' => $channelName,
                'socket_id' => $socketId,
            ]);

            // Check if user is authorized for this channel
            $isAuthorized = $this->authorizeChannel($user, $channelName);

            if (!$isAuthorized) {
                Log::warning('[BROADCAST AUTH] User not authorized for channel', [
                    'user_id' => $user->id,
                    'channel_name' => $channelName,
                ]);
                return response()->json(['error' => 'Forbidden. Access denied.'], 403);
            }

            // 1. Préparer les options pour le constructeur Pusher
            $options = [
                'cluster' => config('broadcasting.connections.pusher.options.cluster'),
                'useTLS' => config('broadcasting.connections.pusher.options.useTLS', true),
            ];

            // Ajouter la clé de chiffrement si elle est définie
            $encryptionKey = config('broadcasting.connections.pusher.options.encryption_master_key_base64');
            if ($encryptionKey) {
                $options['encryption_master_key_base64'] = $encryptionKey;
            }

            // 2. Initialiser Pusher avec toutes les options
            $pusher = new Pusher(
                config('broadcasting.connections.pusher.key'),
                config('broadcasting.connections.pusher.secret'),
                config('broadcasting.connections.pusher.app_id'),
                $options
            );

            $auth = null;
            $logMessage = '';

            // =======================================================
            // ===== CORRECTION ERREUR 500 (undefined method) =====
            // =======================================================
            //
            // Dans la v7.2 de Pusher, authorizeEncryptedChannel() n'existe plus.
            // On utilise authorizeChannel() pour tout.
            // Il inclura le shared_secret automatiquement si la $encryptionKey
            // a été passée au constructeur de $pusher ET si le canal
            // commence par "private-encrypted-" ou "presence-encrypted-".
            //

            if (str_starts_with($channelName, 'presence-')) {
                // Préparer les données utilisateur pour les canaux presence
                $userData = [
                    'user_id' => $user->id,
                    'user_info' => [
                        'id' => $user->id,
                        'name' => $user->display_name,
                        'email' => $user->email,
                    ],
                ];

                // On utilise TOUJOURS authorizeChannel.
                // Pusher v7.2 attend les données "presence" en JSON string.
                $channelData = json_encode($userData);
                $auth = $pusher->authorizeChannel($channelName, $socketId, $channelData);
                $logMessage = '[BROADCAST AUTH] Presence channel authorized';

            } else {
                // C'est un canal "private" normal
                // On utilise TOUJOURS authorizeChannel.
                $auth = $pusher->authorizeChannel($channelName, $socketId);
                $logMessage = '[BROADCAST AUTH] Private channel authorized';
            }
            // =======================================================
            // ===== FIN DE LA CORRECTION ERREUR 500         =====
            // =======================================================

            // Loguer la réponse qui sera envoyée
            Log::info($logMessage, [
                'user_id' => $user->id,
                'channel_name' => $channelName,
                'is_encrypted' => !empty($encryptionKey),
                'response_body_string' => $auth
            ]);

            // Renvoyer une VRAIE réponse JSON
            // La bibliothèque Pusher renvoie une chaîne JSON.
            // Nous devons la décoder en tableau PHP pour que response()->json()
            // puisse la ré-encoder correctement avec le bon Content-Type.
            return response()->json(json_decode($auth, true));

        } catch (\Exception $e) {
            Log::error('[BROADCAST AUTH] Error: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString(),
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'request_data' => $request->all(),
            ]);
            return response()->json([
                'error' => 'Internal server error.',
                'message' => config('app.debug') ? $e->getMessage() : 'An error occurred.',
            ], 500);
        }
    }

    /**
     * Authorize user access to a channel
     */
    protected function authorizeChannel(User $user, string $channelName): bool
    {
        // Handle private-battle.{battleId} channels
        if (preg_match('/^private-battle\.(\d+)$/', $channelName, $matches)) {
            $battleId = $matches[1];

            try {
                $battle = Battle::findOrFail($battleId);
            } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
                Log::warning('[BROADCAST AUTH] Battle not found', [
                    'battle_id' => $battleId,
                    'channel_name' => $channelName,
                ]);
                return false;
            }

            // Correction de l'erreur 403 : Utiliser == au lieu de ===
            // car les IDs peuvent être de types différents (string vs int)
            $isAuthorized = $battle->player1_id == $user->id || $battle->player2_id == $user->id;

            if ($isAuthorized) {
                Log::info('[BROADCAST AUTH] User authorized for battle channel', [
                    'user_id' => $user->id,
                    'battle_id' => $battle->id,
                    'channel_name' => $channelName,
                ]);
            } else {
                Log::warning('[BROADCAST AUTH] User not authorized for battle channel', [
                    'user_id' => $user->id,
                    'battle_id' => $battle->id,
                    'player1_id' => $battle->player1_id,
                    'player2_id' => $battle->player2_id,
                    'channel_name' => $channelName,
                ]);
            }

            return $isAuthorized;
        }

        // Add other channel authorization logic here
        // For now, deny access to unknown channels
        Log::warning('[BROADCAST AUTH] Unknown channel type', [
            'channel_name' => $channelName,
        ]);
        return false;
    }
}
