<?php

namespace App\Services;

// SUPPRIMER TOUS LES IMPORTS D'ÉVÉNMENTS
use App\Models\Battle;
use App\Models\User;
use Illuminate\Support\Str;
use Carbon\Carbon;

class BattleService
{
    // CORRECTION: Suppression du constructeur pour briser la boucle de récursion
    // protected MatchmakingService $matchmakingService;
    // public function __construct(MatchmakingService $matchmakingService)
    // {
    //     $this->matchmakingService = $matchmakingService;
    // }

    /**
     * Create a new battle
     */
    public function createBattle(User $user, string $mode = 'ranked', ?string $roomId = null): Battle
    {
        $roomId = $roomId ?? ($mode === 'friendly' ? $this->generateRoomId() : null);
        
        $battle = Battle::create([
            'room_id' => $roomId,
            'mode' => $mode,
            'status' => 'waiting',
            'player1_id' => $user->id,
            'player1_score' => 0,
            'player2_score' => 0,
            'player1_question_index' => 0,
            'player2_question_index' => 0,
            'player1_answered_questions' => [],
            'player2_answered_questions' => [],
            'total_time_limit' => 300, // 5 minutes
        ]);

        return $battle->load('player1'); // Retourne la bataille avec le joueur 1
    }

    /**
     * Join a battle
     */
    public function joinBattle(Battle $battle, User $user): Battle
    {
        if ($battle->status !== 'waiting') {
            throw new \Exception('Battle is not waiting for players');
        }

        if ($battle->player1_id == $user->id) { // '==' est plus sûr
            throw new \Exception('You cannot join your own battle');
        }

        if ($battle->player2_id !== null) {
            throw new \Exception('Battle is already full');
        }

        $questions = $this->generateQuestions($battle->id);

        $battle->update([
            'player2_id' => $user->id,
            'status' => 'active',
            'questions' => $questions,
            'start_time' => Carbon::now(),
        ]);
        
        // Tous les événements sont supprimés
        return $battle->load(['player1', 'player2']); // Retourne la bataille complète
    }

    /**
     * Update score
     */
    public function updateScore(Battle $battle, User $user, int $questionIndex, bool $isCorrect): void
    {
        if ($battle->status !== 'active') {
            throw new \Exception('Battle is not active');
        }

        $isPlayer1 = $battle->player1_id == $user->id;
        $isPlayer2 = $battle->player2_id == $user->id;

        if (!$isPlayer1 && !$isPlayer2) {
            throw new \Exception('User is not a participant in this battle');
        }

        $answeredQuestions = $isPlayer1 
            ? $battle->player1_answered_questions ?? [] 
            : $battle->player2_answered_questions ?? [];

        if (in_array($questionIndex, $answeredQuestions)) {
            throw new \Exception('Question already answered');
        }

        $answeredQuestions[] = $questionIndex;

        $updateData = [];
        if ($isPlayer1) {
            $updateData['player1_question_index'] = $questionIndex + 1;
            $updateData['player1_answered_questions'] = $answeredQuestions;
            if ($isCorrect) {
                $updateData['player1_score'] = $battle->player1_score + 1;
            }
        } else {
            $updateData['player2_question_index'] = $questionIndex + 1;
            $updateData['player2_answered_questions'] = $answeredQuestions;
            if ($isCorrect) {
                $updateData['player2_score'] = $battle->player2_score + 1;
            }
        }

        $battle->update($updateData);
        $this->checkBattleEnd($battle);
    }

    /**
     * Abandon battle
     */
    public function abandonBattle(Battle $battle, User $user): void
    {
        if ($battle->status !== 'active' && $battle->status !== 'waiting') {
            throw new \Exception('Battle cannot be abandoned');
        }

        $isPlayer1 = $battle->player1_id == $user->id;
        $isPlayer2 = $battle->player2_id == $user->id;

        if (!$isPlayer1 && !$isPlayer2) {
            throw new \Exception('User is not a participant in this battle');
        }

        if ($isPlayer1) {
            $battle->update(['player1_abandoned' => true]);
        } else {
            $battle->update(['player2_abandoned' => true]);
        }
        
        if ($battle->status === 'active') {
            $this->finishBattle($battle);
        }
    }

    /**
     * Finish battle
     */
    public function finishBattle(Battle $battle): void
    {
        if ($battle->status === 'finished') {
            return;
        }

        $winnerId = null;
        $result = 'draw';

        if ($battle->player1_abandoned) {
            $winnerId = $battle->player2_id;
            $result = 'player2_win';
        } elseif ($battle->player2_abandoned) {
            $winnerId = $battle->player1_id;
            $result = 'player1_win';
        } else {
            if ($battle->player1_score > $battle->player2_score) {
                $winnerId = $battle->player1_id;
                $result = 'player1_win';
            } elseif ($battle->player2_score > $battle->player1_score) {
                $winnerId = $battle->player2_id;
                $result = 'player2_win';
            }
        }

        $battle->update([
            'status' => 'finished',
            'winner_id' => $winnerId,
            'result' => $result,
            'end_time' => Carbon::now(),
        ]);

        if ($battle->mode === 'ranked' && $winnerId) {
            $this->updateTrophies($battle, $winnerId);
        }

        dispatch(new \App\Jobs\UpdateLeaderboard());
    }

    /**
     * Check if battle should end
     */
    protected function checkBattleEnd(Battle $battle): void
    {
        $now = Carbon::now();
        $startTime = Carbon::parse($battle->start_time);
        $elapsed = $now->diffInSeconds($startTime);

        if ($elapsed >= $battle->total_time_limit) {
            $this->finishBattle($battle);
        }
    }

    /**
     * Update trophies
     */
    protected function updateTrophies(Battle $battle, int $winnerId): void
    {
        $winner = User::find($winnerId);
        $loserId = $battle->player1_id == $winnerId ? $battle->player2_id : $battle->player1_id;
        $loser = User::find($loserId);

        if ($winner) {
            $winner->increment('trophies', 10);
        }

        if ($loser) {
            $loser->decrement('trophies', 5);
            if ($loser->trophies < 0) {
                $loser->update(['trophies' => 0]);
            }
        }
    }

    /**
     * Generate room ID for friendly battles
     */
    protected function generateRoomId(): string
    {
        return Str::upper(Str::random(6));
    }

    /**
     * Generate questions deterministically based on battle ID
     */
    public function generateQuestions(int $battleId): array
    {
        $questions = range(1, 90); 
        mt_srand($battleId); 
        shuffle($questions);
        mt_srand(); 
        return $questions;
    }
}