<?php

namespace App\Services;

use App\Models\User;
use Kreait\Firebase\Auth;
use Kreait\Firebase\Factory;
use Kreait\Firebase\Exception\Auth\FailedToVerifyToken;

class FirebaseAuthService
{
    protected Auth $auth;

    public function __construct()
    {
        $credentialsPath = config('firebase.credentials_path');
        
        // Si le chemin est relatif, le convertir en chemin absolu
        if ($credentialsPath && !str_starts_with($credentialsPath, '/') && !preg_match('/^[A-Za-z]:\\\\/', $credentialsPath)) {
            // Chemin relatif détecté, utiliser storage_path() pour obtenir le chemin absolu
            $credentialsPath = storage_path('app/firebase-credentials.json');
        }
        
        // Si pas de configuration, utiliser le chemin par défaut
        if (!$credentialsPath) {
            $credentialsPath = storage_path('app/firebase-credentials.json');
        }
        
        // Normaliser les séparateurs de chemin pour Windows
        $credentialsPath = str_replace(['/', '\\'], DIRECTORY_SEPARATOR, $credentialsPath);
        
        // Vérifier si le fichier existe (essayer d'abord le chemin tel quel, puis avec realpath)
        $realPath = realpath($credentialsPath);
        if (!$realPath && !file_exists($credentialsPath)) {
            $suggestedPath = storage_path('app/firebase-credentials.json');
            
            throw new \RuntimeException(
                "Firebase credentials file not found at: {$credentialsPath}\n" .
                "Real path check: " . ($realPath ?: 'NOT FOUND') . "\n" .
                "Please:\n" .
                "1. Download the credentials JSON file from Firebase Console\n" .
                "2. Place it at: {$suggestedPath}\n" .
                "3. Or configure the path in config/firebase.php or .env file\n" .
                "See FIREBASE_SETUP.md for detailed instructions."
            );
        }
        
        // Utiliser realpath si disponible, sinon le chemin original
        $finalPath = $realPath ?: $credentialsPath;

        $factory = (new Factory)->withServiceAccount($finalPath);
        $this->auth = $factory->createAuth();
    }

    /**
     * Verify Firebase token and get user UID
     */
    public function verifyToken(string $token): ?string
    {
        try {
            $verifiedToken = $this->auth->verifyIdToken($token);
            return $verifiedToken->claims()->get('sub'); // UID
        } catch (FailedToVerifyToken $e) {
            return null;
        }
    }

    /**
     * Get user data from Firebase
     */
    public function getUserData(string $uid): ?array
    {
        try {
            $userRecord = $this->auth->getUser($uid);
            return [
                'firebase_uid' => $userRecord->uid,
                'display_name' => $userRecord->displayName,
                'email' => $userRecord->email,
                'photo_url' => $userRecord->photoUrl,
            ];
        } catch (\Exception $e) {
            return null;
        }
    }

    /**
     * Find or create user from Firebase UID
     */
    public function findOrCreateUser(string $uid): ?User
    {
        $user = User::where('firebase_uid', $uid)->first();

        if (!$user) {
            $userData = $this->getUserData($uid);
            if (!$userData) {
                return null;
            }

            $user = User::create([
                'firebase_uid' => $userData['firebase_uid'],
                'display_name' => $userData['display_name'] ?? 'User',
                'email' => $userData['email'],
                'photo_url' => $userData['photo_url'],
                'points' => 500,
                'trophies' => 0,
                'completed_levels' => 0,
            ]);
        }

        return $user;
    }
}

