<?php

namespace App\Services;

use App\Models\User;
use App\Models\UserData;
use App\Models\LevelProgress;

class SyncService
{
    /**
     * Sync user data from client to server
     */
    public function syncToServer(User $user, array $data): void
    {
        // Update user basic info
        if (isset($data['points'])) {
            $user->points = $data['points'];
        }
        if (isset($data['trophies'])) {
            $user->trophies = $data['trophies'];
        }
        if (isset($data['completed_levels'])) {
            $user->completed_levels = $data['completed_levels'];
        }
        $user->last_sync_at = now();
        $user->save();

        // Update user data
        if (isset($data['stats']) || isset($data['level_progress']) || isset($data['unlocked_hints'])) {
            UserData::updateOrCreate(
                ['user_id' => $user->id],
                [
                    'stats' => $data['stats'] ?? null,
                    'level_progress' => $data['level_progress'] ?? null,
                    'unlocked_hints' => $data['unlocked_hints'] ?? null,
                ]
            );
        }

        // Update level progress
        if (isset($data['level_progress']) && is_array($data['level_progress'])) {
            foreach ($data['level_progress'] as $levelData) {
                LevelProgress::updateOrCreate(
                    [
                        'user_id' => $user->id,
                        'level_id' => $levelData['level_id'],
                    ],
                    [
                        'is_completed' => $levelData['is_completed'] ?? false,
                        'best_time' => $levelData['best_time'] ?? null,
                        'attempts' => $levelData['attempts'] ?? 0,
                        'last_played_at' => isset($levelData['last_played_at']) 
                            ? \Carbon\Carbon::parse($levelData['last_played_at']) 
                            : now(),
                    ]
                );
            }
        }
    }

    /**
     * Get user data from server
     */
    public function syncFromServer(User $user): array
    {
        $userData = $user->userData;
        $levelProgress = $user->levelProgress;

        return [
            'user' => [
                'id' => $user->id,
                'firebase_uid' => $user->firebase_uid,
                'display_name' => $user->display_name,
                'email' => $user->email,
                'photo_url' => $user->photo_url,
                'points' => $user->points,
                'trophies' => $user->trophies,
                'completed_levels' => $user->completed_levels,
                'last_sync_at' => $user->last_sync_at,
            ],
            'stats' => $userData?->stats ?? null,
            'level_progress' => $levelProgress->map(function ($progress) {
                return [
                    'level_id' => $progress->level_id,
                    'is_completed' => $progress->is_completed,
                    'best_time' => $progress->best_time,
                    'attempts' => $progress->attempts,
                    'last_played_at' => $progress->last_played_at,
                ];
            })->toArray(),
            'unlocked_hints' => $userData?->unlocked_hints ?? null,
        ];
    }

    /**
     * Sync level progress
     */
    public function syncLevelProgress(User $user, array $levelData): void
    {
        LevelProgress::updateOrCreate(
            [
                'user_id' => $user->id,
                'level_id' => $levelData['level_id'],
            ],
            [
                'is_completed' => $levelData['is_completed'] ?? false,
                'best_time' => $levelData['best_time'] ?? null,
                'attempts' => $levelData['attempts'] ?? 0,
                'last_played_at' => isset($levelData['last_played_at']) 
                    ? \Carbon\Carbon::parse($levelData['last_played_at']) 
                    : now(),
            ]
        );
    }
}

