<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Obtenir Token Firebase - Cadenas</title>
    <!-- Firebase SDK -->
    <script src="https://www.gstatic.com/firebasejs/10.7.1/firebase-app-compat.js"></script>
    <script src="https://www.gstatic.com/firebasejs/10.7.1/firebase-auth-compat.js"></script>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            display: flex;
            justify-content: center;
            align-items: center;
            padding: 20px;
        }

        .container {
            background: white;
            border-radius: 20px;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
            max-width: 600px;
            width: 100%;
            padding: 40px;
        }

        h1 {
            color: #333;
            margin-bottom: 10px;
            font-size: 28px;
        }

        .subtitle {
            color: #666;
            margin-bottom: 30px;
            font-size: 14px;
        }

        .form-group {
            margin-bottom: 20px;
        }

        label {
            display: block;
            margin-bottom: 8px;
            color: #333;
            font-weight: 600;
            font-size: 14px;
        }

        input[type="email"],
        input[type="password"],
        input[type="text"] {
            width: 100%;
            padding: 12px;
            border: 2px solid #e0e0e0;
            border-radius: 8px;
            font-size: 14px;
            transition: border-color 0.3s;
        }

        input:focus {
            outline: none;
            border-color: #667eea;
        }

        button {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border: none;
            padding: 12px 30px;
            border-radius: 8px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            width: 100%;
            transition: transform 0.2s, box-shadow 0.2s;
        }

        button:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(102, 126, 234, 0.4);
        }

        button:disabled {
            opacity: 0.6;
            cursor: not-allowed;
            transform: none;
        }

        .result {
            margin-top: 20px;
            padding: 15px;
            border-radius: 8px;
            display: none;
        }

        .result.success {
            background: #d4edda;
            border: 2px solid #28a745;
            color: #155724;
        }

        .result.error {
            background: #f8d7da;
            border: 2px solid #dc3545;
            color: #721c24;
        }

        .result pre {
            margin: 10px 0 0 0;
            white-space: pre-wrap;
            word-wrap: break-word;
            font-size: 12px;
            max-height: 300px;
            overflow-y: auto;
            background: rgba(0,0,0,0.1);
            padding: 10px;
            border-radius: 5px;
        }

        .loading {
            display: inline-block;
            width: 16px;
            height: 16px;
            border: 3px solid rgba(255, 255, 255, 0.3);
            border-radius: 50%;
            border-top-color: white;
            animation: spin 1s linear infinite;
            margin-right: 8px;
            vertical-align: middle;
        }

        @keyframes spin {
            to { transform: rotate(360deg); }
        }

        .info-box {
            background: #d1ecf1;
            border: 2px solid #17a2b8;
            border-radius: 8px;
            padding: 15px;
            margin-bottom: 30px;
            color: #0c5460;
        }

        .info-box h3 {
            margin-bottom: 10px;
            color: #0c5460;
        }

        .copy-btn {
            background: #28a745;
            margin-top: 10px;
            padding: 8px 20px;
            font-size: 14px;
        }

        .copy-btn:hover {
            background: #218838;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>🔑 Obtenir Token Firebase</h1>
        <p class="subtitle">Connectez-vous pour obtenir un token Firebase de test</p>

        <div class="info-box">
            <h3>ℹ️ Instructions</h3>
            <p>1. Configurez votre Firebase Web API Key ci-dessous</p>
            <p>2. Connectez-vous avec un compte existant ou créez un nouveau compte</p>
            <p>3. Copiez le token généré et utilisez-le dans la page de test</p>
        </div>

        <!-- Configuration Firebase -->
        <div class="form-group">
            <label for="firebase-api-key">Firebase Web API Key :</label>
            <input type="text" id="firebase-api-key" placeholder="Votre Firebase Web API Key">
            <small style="color: #666; font-size: 12px;">
                Trouvez-la dans Firebase Console > Project Settings > Web App > API Key
            </small>
        </div>

        <!-- Formulaire de connexion -->
        <div id="login-form">
            <div class="form-group">
                <label for="email">Email :</label>
                <input type="email" id="email" placeholder="votre@email.com">
            </div>

            <div class="form-group">
                <label for="password">Mot de passe :</label>
                <input type="password" id="password" placeholder="Votre mot de passe">
            </div>

            <button onclick="signIn()" id="btn-signin">Se connecter</button>
            <button onclick="signUp()" id="btn-signup" style="margin-top: 10px; background: #17a2b8;">Créer un compte</button>
        </div>

        <!-- Résultat -->
        <div id="result" class="result"></div>

        <!-- Token affiché -->
        <div id="token-section" style="display: none;">
            <h3 style="margin-top: 20px; color: #333;">🎉 Token Firebase obtenu !</h3>
            <div class="form-group">
                <label for="token-display">Token :</label>
                <textarea id="token-display" readonly style="width: 100%; min-height: 100px; padding: 12px; border: 2px solid #28a745; border-radius: 8px; font-family: monospace; font-size: 12px;"></textarea>
            </div>
            <button onclick="copyToken()" class="copy-btn">📋 Copier le Token</button>
            <button onclick="goToTestPage()" style="margin-top: 10px; background: #667eea;">🧪 Aller à la page de test</button>
        </div>
    </div>

    <script>
        let firebaseApp = null;
        let currentToken = null;

        function getFirebaseConfig() {
            const apiKey = document.getElementById('firebase-api-key').value.trim();
            
            if (!apiKey) {
                alert('Veuillez entrer votre Firebase Web API Key');
                return null;
            }

            // Vous devez également obtenir ces valeurs depuis Firebase Console
            // Pour l'instant, on utilise seulement l'API Key
            // En production, vous devriez stocker ces valeurs de manière sécurisée
            return {
                apiKey: apiKey,
                authDomain: window.location.hostname, // Ajustez selon votre configuration
                // projectId, appId, etc. peuvent être ajoutés si nécessaire
            };
        }

        async function initializeFirebase() {
            const config = getFirebaseConfig();
            if (!config) return false;

            try {
                if (!firebase.apps.length) {
                    firebase.initializeApp(config);
                    firebaseApp = firebase.app();
                }
                return true;
            } catch (error) {
                showResult('error', 'Erreur d\'initialisation Firebase: ' + error.message);
                return false;
            }
        }

        async function signIn() {
            const email = document.getElementById('email').value.trim();
            const password = document.getElementById('password').value.trim();

            if (!email || !password) {
                showResult('error', 'Veuillez remplir tous les champs');
                return;
            }

            const button = document.getElementById('btn-signin');
            button.disabled = true;
            button.innerHTML = '<span class="loading"></span>Connexion...';

            try {
                if (!await initializeFirebase()) {
                    button.disabled = false;
                    button.innerHTML = 'Se connecter';
                    return;
                }

                const userCredential = await firebase.auth().signInWithEmailAndPassword(email, password);
                const token = await userCredential.user.getIdToken();
                
                currentToken = token;
                showToken(token);
                showResult('success', '✅ Connexion réussie !');
            } catch (error) {
                showResult('error', 'Erreur de connexion: ' + error.message);
            } finally {
                button.disabled = false;
                button.innerHTML = 'Se connecter';
            }
        }

        async function signUp() {
            const email = document.getElementById('email').value.trim();
            const password = document.getElementById('password').value.trim();

            if (!email || !password) {
                showResult('error', 'Veuillez remplir tous les champs');
                return;
            }

            if (password.length < 6) {
                showResult('error', 'Le mot de passe doit contenir au moins 6 caractères');
                return;
            }

            const button = document.getElementById('btn-signup');
            button.disabled = true;
            button.innerHTML = '<span class="loading"></span>Création du compte...';

            try {
                if (!await initializeFirebase()) {
                    button.disabled = false;
                    button.innerHTML = 'Créer un compte';
                    return;
                }

                const userCredential = await firebase.auth().createUserWithEmailAndPassword(email, password);
                const token = await userCredential.user.getIdToken();
                
                currentToken = token;
                showToken(token);
                showResult('success', '✅ Compte créé avec succès !');
            } catch (error) {
                showResult('error', 'Erreur de création: ' + error.message);
            } finally {
                button.disabled = false;
                button.innerHTML = 'Créer un compte';
            }
        }

        function showToken(token) {
            document.getElementById('token-display').value = token;
            document.getElementById('token-section').style.display = 'block';
            document.getElementById('login-form').style.display = 'none';
        }

        function copyToken() {
            const tokenDisplay = document.getElementById('token-display');
            tokenDisplay.select();
            document.execCommand('copy');
            showResult('success', '✅ Token copié dans le presse-papier !');
        }

        function goToTestPage() {
            if (currentToken) {
                window.location.href = `/test-firebase-page?token=${encodeURIComponent(currentToken)}`;
            }
        }

        function showResult(type, message) {
            const resultDiv = document.getElementById('result');
            resultDiv.className = `result ${type}`;
            resultDiv.style.display = 'block';
            resultDiv.innerHTML = `<strong>${message}</strong>`;
            
            setTimeout(() => {
                if (type !== 'error') {
                    resultDiv.style.display = 'none';
                }
            }, 5000);
        }

        // Charger le token depuis l'URL si présent
        window.addEventListener('load', () => {
            const urlParams = new URLSearchParams(window.location.search);
            const token = urlParams.get('token');
            if (token) {
                showToken(token);
            }
        });
    </script>
</body>
</html>

