<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Test Firebase Auth - Cadenas</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            display: flex;
            justify-content: center;
            align-items: center;
            padding: 20px;
        }

        .container {
            background: white;
            border-radius: 20px;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
            max-width: 800px;
            width: 100%;
            padding: 40px;
        }

        h1 {
            color: #333;
            margin-bottom: 10px;
            font-size: 28px;
        }

        .subtitle {
            color: #666;
            margin-bottom: 30px;
            font-size: 14px;
        }

        .test-section {
            margin-bottom: 30px;
            padding: 20px;
            background: #f8f9fa;
            border-radius: 10px;
        }

        .test-section h2 {
            color: #333;
            font-size: 20px;
            margin-bottom: 15px;
        }

        .form-group {
            margin-bottom: 20px;
        }

        label {
            display: block;
            margin-bottom: 8px;
            color: #333;
            font-weight: 600;
            font-size: 14px;
        }

        input[type="text"],
        textarea {
            width: 100%;
            padding: 12px;
            border: 2px solid #e0e0e0;
            border-radius: 8px;
            font-size: 14px;
            font-family: 'Courier New', monospace;
            transition: border-color 0.3s;
        }

        input[type="text"]:focus,
        textarea:focus {
            outline: none;
            border-color: #667eea;
        }

        textarea {
            min-height: 100px;
            resize: vertical;
        }

        button {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border: none;
            padding: 12px 30px;
            border-radius: 8px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: transform 0.2s, box-shadow 0.2s;
        }

        button:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(102, 126, 234, 0.4);
        }

        button:active {
            transform: translateY(0);
        }

        button:disabled {
            opacity: 0.6;
            cursor: not-allowed;
            transform: none;
        }

        .result {
            margin-top: 20px;
            padding: 15px;
            border-radius: 8px;
            display: none;
        }

        .result.success {
            background: #d4edda;
            border: 2px solid #28a745;
            color: #155724;
        }

        .result.error {
            background: #f8d7da;
            border: 2px solid #dc3545;
            color: #721c24;
        }

        .result.info {
            background: #d1ecf1;
            border: 2px solid #17a2b8;
            color: #0c5460;
        }

        .result pre {
            margin: 10px 0 0 0;
            white-space: pre-wrap;
            word-wrap: break-word;
            font-size: 12px;
            max-height: 400px;
            overflow-y: auto;
        }

        .loading {
            display: inline-block;
            width: 16px;
            height: 16px;
            border: 3px solid rgba(255, 255, 255, 0.3);
            border-radius: 50%;
            border-top-color: white;
            animation: spin 1s linear infinite;
            margin-right: 8px;
            vertical-align: middle;
        }

        @keyframes spin {
            to { transform: rotate(360deg); }
        }

        .status-badge {
            display: inline-block;
            padding: 4px 12px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
            margin-left: 10px;
        }

        .status-badge.success {
            background: #28a745;
            color: white;
        }

        .status-badge.error {
            background: #dc3545;
            color: white;
        }

        .instructions {
            background: #fff3cd;
            border: 2px solid #ffc107;
            border-radius: 8px;
            padding: 15px;
            margin-bottom: 30px;
            color: #856404;
        }

        .instructions h3 {
            margin-bottom: 10px;
            color: #856404;
        }

        .instructions ol {
            margin-left: 20px;
        }

        .instructions li {
            margin-bottom: 5px;
        }

        code {
            background: #f4f4f4;
            padding: 2px 6px;
            border-radius: 4px;
            font-family: 'Courier New', monospace;
            font-size: 12px;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>🔐 Test Firebase Authentication</h1>
        <p class="subtitle">Testez l'authentification Firebase directement depuis votre navigateur</p>

        <div class="instructions">
            <h3>📋 Comment obtenir un Token Firebase ?</h3>
            <p><strong>Méthode 1 : Depuis votre application mobile (Recommandé)</strong></p>
            <ol>
                <li><strong>Flutter :</strong> Ajoutez ce code dans votre app :
                    <pre style="background: #f4f4f4; padding: 10px; border-radius: 5px; margin: 10px 0; overflow-x: auto;"><code>User? user = FirebaseAuth.instance.currentUser;
String? token = await user?.getIdToken();
print('Token: $token');</code></pre>
                </li>
                <li><strong>iOS (Swift) :</strong>
                    <pre style="background: #f4f4f4; padding: 10px; border-radius: 5px; margin: 10px 0; overflow-x: auto;"><code>Auth.auth().currentUser?.getIDToken { token, error in
    if let token = token {
        print("Token: \(token)")
    }
}</code></pre>
                </li>
                <li><strong>Android (Kotlin) :</strong>
                    <pre style="background: #f4f4f4; padding: 10px; border-radius: 5px; margin: 10px 0; overflow-x: auto;"><code>FirebaseAuth.getInstance().currentUser?.getIdToken(true)
    ?.addOnSuccessListener { result ->
        val token = result.token
        Log.d("Firebase", "Token: $token")
    }</code></pre>
                </li>
            </ol>
            <p><strong>Méthode 2 : Créer un utilisateur de test</strong></p>
            <ol>
                <li>Allez sur <a href="https://console.firebase.google.com/" target="_blank">Firebase Console</a></li>
                <li>Authentication > Users > Add user</li>
                <li>Créez un utilisateur avec email/mot de passe</li>
                <li>Connectez-vous dans votre app et récupérez le token</li>
            </ol>
            <p style="margin-top: 15px;"><strong>💡 Solution rapide :</strong> <a href="/get-firebase-token" style="color: #667eea; font-weight: bold; text-decoration: underline;">Cliquez ici pour obtenir un token directement depuis cette page</a></p>
            <p style="margin-top: 10px;"><strong>📖 Documentation :</strong> Consultez <code>GET_TOKEN_GUIDE.md</code> pour plus de détails.</p>
        </div>

        <!-- Lien rapide pour obtenir un token -->
        <div style="background: #e7f3ff; border: 2px solid #2196F3; border-radius: 8px; padding: 20px; margin-bottom: 30px; text-align: center;">
            <h3 style="color: #1976D2; margin-bottom: 10px;">🚀 Obtenir un Token Maintenant</h3>
            <p style="color: #666; margin-bottom: 15px;">Connectez-vous directement depuis cette page pour obtenir un token Firebase</p>
            <a href="/get-firebase-token" style="display: inline-block; background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 12px 30px; border-radius: 8px; text-decoration: none; font-weight: 600; transition: transform 0.2s;">🔑 Obtenir un Token Firebase</a>
        </div>

        <!-- Test 1: Configuration Firebase -->
        <div class="test-section">
            <h2>1. Test de Configuration Firebase</h2>
            <p style="color: #666; margin-bottom: 15px; font-size: 14px;">Vérifie que Firebase est correctement configuré</p>
            <button onclick="testConfiguration()" id="btn-config">Tester la Configuration</button>
            <div id="result-config" class="result"></div>
        </div>

        <!-- Test 2: Vérification du Token -->
        <div class="test-section">
            <h2>2. Test de Vérification du Token</h2>
            <p style="color: #666; margin-bottom: 15px; font-size: 14px;">Vérifie qu'un token Firebase est valide</p>
            <div class="form-group">
                <label for="token">Token Firebase :</label>
                <textarea id="token" placeholder="Collez votre token Firebase ici..."></textarea>
            </div>
            <button onclick="testToken()" id="btn-token">Tester le Token</button>
            <div id="result-token" class="result"></div>
        </div>

        <!-- Test 3: API Réelle -->
        <div class="test-section">
            <h2>3. Test de l'API Réelle</h2>
            <p style="color: #666; margin-bottom: 15px; font-size: 14px;">Teste l'endpoint API réel avec authentification</p>
            <button onclick="testAPI()" id="btn-api">Tester l'API /api/user</button>
            <div id="result-api" class="result"></div>
        </div>
    </div>

    <script>
        const API_BASE_URL = window.location.origin;

        function showResult(elementId, type, message, data = null) {
            const element = document.getElementById(elementId);
            element.className = `result ${type}`;
            element.style.display = 'block';
            
            if (data) {
                element.innerHTML = `<strong>${message}</strong><pre>${JSON.stringify(data, null, 2)}</pre>`;
            } else {
                element.innerHTML = `<strong>${message}</strong>`;
            }
        }

        function setLoading(buttonId, loading) {
            const button = document.getElementById(buttonId);
            if (loading) {
                button.disabled = true;
                button.innerHTML = '<span class="loading"></span>Chargement...';
            } else {
                button.disabled = false;
                button.innerHTML = button.getAttribute('data-original-text') || button.textContent;
            }
        }

        async function testConfiguration() {
            const buttonId = 'btn-config';
            const resultId = 'result-config';
            const button = document.getElementById(buttonId);
            button.setAttribute('data-original-text', button.textContent);
            setLoading(buttonId, true);

            try {
                const response = await fetch(`${API_BASE_URL}/test-firebase-auth`);
                const data = await response.json();

                if (data.status === 'success') {
                    showResult(resultId, 'success', '✅ Configuration Firebase OK !', data);
                } else {
                    showResult(resultId, 'error', '❌ Erreur de configuration', data);
                }
            } catch (error) {
                showResult(resultId, 'error', '❌ Erreur : ' + error.message);
            } finally {
                setLoading(buttonId, false);
            }
        }

        async function testToken() {
            const buttonId = 'btn-token';
            const resultId = 'result-token';
            const tokenInput = document.getElementById('token');
            const token = tokenInput.value.trim();

            if (!token) {
                showResult(resultId, 'error', '❌ Veuillez entrer un token Firebase');
                return;
            }

            const button = document.getElementById(buttonId);
            button.setAttribute('data-original-text', button.textContent);
            setLoading(buttonId, true);

            try {
                const response = await fetch(`${API_BASE_URL}/test-firebase-token`, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({ token: token })
                });

                const data = await response.json();

                if (data.status === 'success') {
                    showResult(resultId, 'success', '✅ Token valide !', data);
                } else {
                    showResult(resultId, 'error', '❌ Token invalide ou erreur', data);
                }
            } catch (error) {
                showResult(resultId, 'error', '❌ Erreur : ' + error.message);
            } finally {
                setLoading(buttonId, false);
            }
        }

        async function testAPI() {
            const buttonId = 'btn-api';
            const resultId = 'result-api';
            const tokenInput = document.getElementById('token');
            const token = tokenInput.value.trim();

            if (!token) {
                showResult(resultId, 'error', '❌ Veuillez d\'abord entrer un token Firebase');
                return;
            }

            const button = document.getElementById(buttonId);
            button.setAttribute('data-original-text', button.textContent);
            setLoading(buttonId, true);

            try {
                const response = await fetch(`${API_BASE_URL}/api/user`, {
                    method: 'GET',
                    headers: {
                        'Authorization': `Bearer ${token}`,
                        'Content-Type': 'application/json',
                    }
                });

                const data = await response.json();

                if (response.ok) {
                    showResult(resultId, 'success', '✅ API fonctionne correctement !', data);
                } else {
                    showResult(resultId, 'error', `❌ Erreur API (${response.status})`, data);
                }
            } catch (error) {
                showResult(resultId, 'error', '❌ Erreur : ' + error.message);
            } finally {
                setLoading(buttonId, false);
            }
        }

        // Test automatique de la configuration au chargement
        window.addEventListener('load', () => {
            testConfiguration();
            
            // Charger le token depuis l'URL si présent
            const urlParams = new URLSearchParams(window.location.search);
            const token = urlParams.get('token');
            if (token) {
                document.getElementById('token').value = token;
                // Tester automatiquement le token après 1 seconde
                setTimeout(() => {
                    testToken();
                }, 1000);
            }
        });
    </script>
</body>
</html>

