<?php

use Illuminate\Support\Facades\Broadcast;
use Illuminate\Support\Facades\Log;
use App\Models\Battle;
use App\Models\User;

/**
 * Authorize private channel: private-battle.{battleId}
 * 
 * Laravel automatically prefixes private channels with "private-"
 * So "battle.{battleId}" becomes "private-battle.{battleId}"
 */
Broadcast::channel('battle.{battleId}', function ($user, $battleId) {
    
    // --- DÉBUT DU DEBUG ---
    // Log::info va écrire dans le fichier storage/logs/laravel.log
    Log::info('[AUTH BROADCAST] Tentative d\'accès au canal', [
        'channel' => 'private-battle.' . $battleId,
        'user_object' => $user ? [
            'id' => $user->id,
            'firebase_uid' => $user->firebase_uid,
            'display_name' => $user->display_name,
        ] : null,
        'user_id' => $user ? $user->id : 'Utilisateur NULL',
        'user_type' => gettype($user),
        'battle_id' => $battleId,
        'timestamp' => now()->toDateTimeString(),
    ]);
    // --- FIN DU DEBUG ---
    
    // Si l'utilisateur n'est pas trouvé, refuser explicitement
    if ($user === null) {
        Log::warning('[AUTH BROADCAST] Accès refusé : Utilisateur NULL', [
            'battle_id' => $battleId,
            'channel' => 'private-battle.' . $battleId,
        ]);
        return false;
    }
    
    // Vérifier que $user est une instance de User
    if (!($user instanceof User)) {
        Log::warning('[AUTH BROADCAST] Accès refusé : Type d\'utilisateur invalide', [
            'battle_id' => $battleId,
            'user_type' => gettype($user),
            'user_class' => is_object($user) ? get_class($user) : 'N/A',
        ]);
        return false;
    }

    // Votre logique d'autorisation (qui ne sera atteinte que si $user n'est pas null)
    $battle = Battle::find($battleId);
    
    if (!$battle) {
        Log::warning('[AUTH BROADCAST] Accès refusé : Bataille non trouvée', [
            'battle_id' => $battleId,
            'user_id' => $user->id,
        ]);
        return false;
    }

    // Check if user is a participant in the battle
    $isAuthorized = $battle->player1_id === $user->id || $battle->player2_id === $user->id;
    
    if ($isAuthorized) {
        Log::info('[AUTH BROADCAST] Accès autorisé', [
            'user_id' => $user->id,
            'battle_id' => $battleId,
            'player1_id' => $battle->player1_id,
            'player2_id' => $battle->player2_id,
        ]);
        // Return user data for presence channels, or true for private channels
        // For private channels, we can return true or user data
        return ['id' => $user->id, 'name' => $user->display_name];
    }
    
    Log::warning('[AUTH BROADCAST] Accès refusé : Utilisateur non participant', [
        'user_id' => $user->id,
        'battle_id' => $battleId,
        'player1_id' => $battle->player1_id,
        'player2_id' => $battle->player2_id,
    ]);
    
    return false;
});
